/* UART Events Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/queue.h"
#include "driver/uart.h"
#include "esp_log.h"
#include "portmacro.h"

#define UART_NUM UART_NUM_2
#define TXD_PIN 	17
#define RXD_PIN 	16

#define BUF_SIZE (1024)

static QueueHandle_t event_queue;


typedef struct {
    int len;
    uint8_t *data;
} uart_packet_t;

static QueueHandle_t data_queue;

static void uart_event_task(void *pvParameters)
{
    uart_event_t event;
    for (;;) {
        //Waiting for UART event.
        if (xQueueReceive(event_queue, (void *)&event, (TickType_t)portMAX_DELAY)) {
            switch (event.type) {
            //Event of UART receiving data
            /*We'd better handler data event fast, there would be much more data events than
            other types of events. If we take too much time on data event, the queue might
            be full.*/
            case UART_DATA:
				uart_packet_t pkt;
				pkt.data = malloc(event.size);
    			if (pkt.data == NULL) {
        			ESP_LOGE("UART", "Malloc failed");
       				 break;
   				}
				ESP_LOGI("UART", "Data Event: Bytes=%d", event.size);
				pkt.len = uart_read_bytes(UART_NUM, pkt.data, event.size, portMAX_DELAY);
				xQueueSend(data_queue, &pkt, portMAX_DELAY);
                break;
            //Event of HW FIFO overflow detected
            case UART_FIFO_OVF:
                ESP_LOGE("uart", "hw fifo overflow");
                // If fifo overflow happened, you should consider adding flow control for your application.
                // The ISR has already reset the rx FIFO,
                // As an example, we directly flush the rx buffer here in order to read more data.
                uart_flush_input(UART_NUM);
                xQueueReset(event_queue);
                break;
            //Event of UART ring buffer full
            case UART_BUFFER_FULL:
                ESP_LOGI("uart", "ring buffer full");
                // If buffer full happened, you should consider increasing your buffer size
                // As an example, we directly flush the rx buffer here in order to read more data.
                uart_flush_input(UART_NUM);
                xQueueReset(event_queue);
                break;
            //Event of UART RX break detected
            case UART_BREAK:
                ESP_LOGI("uart", "uart rx break");
                break;
            //Event of UART parity check error
            case UART_PARITY_ERR:
                ESP_LOGI("uart", "uart parity error");
                break;
            //Event of UART frame error
            case UART_FRAME_ERR:
                ESP_LOGI("uart", "uart frame error");
                break;
            //Others
            default:
                ESP_LOGI("uart", "uart event type: %d", event.type);
                break;
            }
        }
    }
}


static void uart_process_task(void *pvParameters)
{
	uart_packet_t pkt;
	for (;;)
	{
		if (xQueueReceive(data_queue, &pkt, portMAX_DELAY))
		{
			ESP_LOGI ("uart", "processing data");
			uart_write_bytes(UART_NUM, pkt.data, pkt.len);
			free(pkt.data);
		}
	}
}



void app_main(void)
{

    /* Configure parameters of an UART driver,
     * communication pins and install the driver */
    uart_config_t uart_config = {
        .baud_rate = 115200,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
        .source_clk = UART_SCLK_DEFAULT,
    };
    //Install UART driver, and get the queue.
    uart_driver_install(UART_NUM, BUF_SIZE * 2, BUF_SIZE, 20, &event_queue, 0);
    uart_param_config(UART_NUM, &uart_config);

    //Set UART pins (using UART0 default pins ie no changes.)
	uart_set_pin(UART_NUM, TXD_PIN, RXD_PIN, UART_PIN_NO_CHANGE,UART_PIN_NO_CHANGE);

	data_queue = xQueueCreate(5, sizeof(uart_packet_t));

	//Create a task to handler UART event from ISR
    xTaskCreate(uart_event_task, "uart_event_task", 3072, NULL, 12, NULL);
	xTaskCreate(uart_process_task, "uart_process_task", 2048, NULL, 11, NULL);
}
