//*****************************************************************************
//
// blinky.c - Simple example to blink the on-board LED.
//
// Copyright (c) 2012-2020 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.2.0.295 of the EK-TM4C123GXL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/pin_map.h"
#include "driverlib/gpio.h"
#include "driverlib/sysctl.h"
#include "driverlib/interrupt.h"
#include "basic_conf.h"
#include "driverlib/adc.h"

#include "driverlib/uart.h"
#include "utils/uartstdio.h"
void UART0_Init(void)
{
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    GPIOPinConfigure(GPIO_PA0_U0RX);
    GPIOPinConfigure(GPIO_PA1_U0TX);
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    UARTClockSourceSet(UART0_BASE, UART_CLOCK_PIOSC);
    
    UARTStdioConfig(0, 115200, 16000000);
}

void ADC0_Init(void)
{
    //
    // 1. Enable ADC0 and GPIOE clocks
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_ADC0));
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOE));

    //
    // 2. Configure PE3 (AIN0), PE2 (AIN1), PE1 (AIN2) & PE0 (AIN4) as analog input
    //
    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_3|GPIO_PIN_2|GPIO_PIN_1|GPIO_PIN_0);

    //
    // 3. Configure ADC0 Sequencer 2 for processor trigger
    //
    ADCSequenceDisable(ADC0_BASE, 2);  // Disable SS3 before config
    ADCSequenceConfigure(ADC0_BASE, 2, ADC_TRIGGER_PROCESSOR, 0);

    //
    // 4. Configure step 0 of SS2 to sample AIN0
    //
    ADCSequenceStepConfigure(ADC0_BASE, 2, 0, ADC_CTL_CH0);
    //  Configure step 1 of SS2 to sample AIN1
    ADCSequenceStepConfigure(ADC0_BASE, 2, 1, ADC_CTL_CH1);
    //  Configure step 2 of SS2 to sample AIN2
    ADCSequenceStepConfigure(ADC0_BASE, 2, 2, ADC_CTL_CH2);
    //  Configure step 3 of SS2 to sample AIN3
    ADCSequenceStepConfigure(ADC0_BASE, 2, 3, ADC_CTL_CH3|ADC_CTL_END);

    //
    // 5. Enable SS2
    //
    ADCSequenceEnable(ADC0_BASE, 2);
}

void ADC0_Read(uint32_t *array)
{
    // Trigger SS2 conversion
    ADCProcessorTrigger(ADC0_BASE, 2);

    // Read the ADC value
    ADCSequenceDataGet(ADC0_BASE, 2, array);
}

uint32_t adcVal[4];

char val1Array[10];

int main(void)
{
    systemClockConfig();
    UART0_Init();
    ADC0_Init();
    IntMasterEnable();
    while(1)
    {
        ADC0_Read(adcVal);
        int i=0;
        for (i=0; i<4; i++){
            float_to_string(adcVal[i], val1Array, 0);
            UARTprintf("CH%d: %u\t", i, adcVal[i]);
        }
        UARTprintf("\n");
        delay_ms(200);
    }
}
