#include <Adafruit_GFX.h>
#include <Adafruit_ILI9341.h>
#include <XPT2046_Touchscreen.h>
#include <SoftwareSerial.h>
#include "DFRobotDFPlayerMini.h"

// ----------------- Pins -----------------
#define TFT_CS 10
#define TFT_DC 9
#define TFT_RST 8
#define TS_CS 6

#define DF_RX 4   // Arduino reads from DFPlayer TX (SoftwareSerial RX pin)
#define DF_TX 5   // Arduino writes to DFPlayer RX (SoftwareSerial TX pin)

// Touch calibration (adjust if needed)
int TS_MINX = 350;
int TS_MAXX = 3549;
int TS_MINY = 505;
int TS_MAXY = 3795;

// Screen size (portrait)
const int SCR_W = 240;
const int SCR_H = 320;
char tft_buf[32];

// ----------------- Objects -----------------
Adafruit_ILI9341 tft = Adafruit_ILI9341(TFT_CS, TFT_DC, TFT_RST);
XPT2046_Touchscreen ts(TS_CS);
SoftwareSerial dfSerial(DF_RX, DF_TX); // NOTE: first arg is RX pin for Arduino
DFRobotDFPlayerMini myDFPlayer;

// ----------------- Tracks -----------------
String trackList[] = {"Track 01","Track 02","Track 03","Track 04","Track 05"};
int totalTracks = 5;
int currentTrack = 0;

// ----------------- UI Variables -----------------
enum Screen {PLAYER, PLAYLIST, SETTINGS};
Screen currentScreen = PLAYLIST;

bool isPlaying = false;
int elapsedTime = 0;
int trackDuration = 40; // in seconds (example)
int titleX = 4;         // not used for scrolling (static title)
int volumeLevel = 20;    // 0-30
int pulseRadius = 10;
bool growing = true;
int currentTheme = 0;
int brightnessLevel = 200;
int currentEQ = 0;

// --- Button geometry (portrait layout) ---
const int titleBarH = 32;
const int eqAreaTop = 50;
const int eqAreaBottom = 180;

const int btnSize = 48;
int btnY = 200;                // vertical position of prev/play/next buttons
int prevX = 40;
int playX = 96;
int nextX = 152;

const int volX = 20;
const int volY = 270;
const int volW = 200;
const int volH = 12;

const int playlistBtnX = SCR_W-20;   // small home/playlist button top-left
const int playlistBtnY = 2;
const int playlistBtnW = 20;
const int playlistBtnH = 20;

// ----------------- Double-buffer-ish state for partial updates -----------------
int prevBarWidth = 0;
int prevEQHeights[5] = {0,0,0,0,0};

// ----------------- Setup -----------------
void setup() {
  Serial.begin(9600);
  dfSerial.begin(9600);

  tft.begin();
  ts.begin();
  tft.setRotation(2);   // TFT rotation: 0 => width=240, height=320
  ts.setRotation(0);    // Touch rotation must match TFT to keep coordinates aligned

  if(!myDFPlayer.begin(dfSerial)){
    Serial.println("DFPlayer error!");
  } else {
    Serial.println("DFPlayer OK");
    myDFPlayer.volume(volumeLevel);
  }

  drawPlaylistScreen(); // start with playlist
}

// ----------------- Loop -----------------
void loop() {
  // Use timed partial updates to avoid full-screen redraw and flicker
  static unsigned long lastTouchCheck = 0;
  static unsigned long lastEQ = 0;
  static unsigned long lastProgress = 0;
  static unsigned long lastUIFast = 0;

  unsigned long now = millis();

  // Poll touch frequently
  if (now - lastTouchCheck >= 30) {
    lastTouchCheck = now;
    updateScreen();
  }

  // Animate equalizer ~9 FPS
  if (currentScreen == PLAYER && now - lastEQ >= 110) {
    lastEQ = now;
    updateEqualizer();   // partial redraw of eq bars
  }

  // Update progress bar at 4-10 FPS for smooth visual but not heavy
  if (currentScreen == PLAYER && now - lastProgress >= 150) {
    lastProgress = now;
    updateProgressBar(); // partial redraw of progress only
  }

  // Update smaller UI bits occasionally (play/pause, nav, volume)
  // if (currentScreen == PLAYER && now - lastUIFast >= 250) {
  //   lastUIFast = now;
  //   drawPlayPauseButton(); // partial redraw area only
  //   drawPrevButton();
  //   drawNextButton();
  //   drawVolumeSlider();
  //   // title is static (no scrolling), no need to redraw
  // }

  // Only advance simulated track elapsed time if playing, at 1s granularity
  // (This is kept separate — it doesn't force a full redraw)
  static unsigned long lastSec = 0;
  if (isPlaying && (now - lastSec >= 1000)) {
    lastSec = now;
    elapsedTime++;
    if (elapsedTime > trackDuration) {
      elapsedTime = 0;
      nextTrack();
    }
    // we intentionally do NOT update time display every second to avoid flicker
    // If you want it, call updateProgressBar() and drawTime() here (but you said time counter No)
  }

  // small yield
  delay(100);
}

// ----------------- Screen Updates -----------------
void updateScreen(){
  if(ts.touched()){
    TS_Point p = ts.getPoint();
    // Map raw touch to screen coordinates (portrait)
    int tx = map(p.x, TS_MINX, TS_MAXX, 0, SCR_W);
    int ty = map(p.y, TS_MINY, TS_MAXY, 0, SCR_H);

    // bounds clamp
    tx = constrain(tx, 0, SCR_W);
    ty = constrain(ty, 0, SCR_H);

    switch(currentScreen){
      case PLAYER:
        // check top-left playlist/home button
        if(tx >= playlistBtnX && tx <= playlistBtnX + playlistBtnW && ty >= playlistBtnY && ty <= playlistBtnY + playlistBtnH){
          currentScreen = PLAYLIST;
          drawPlaylistScreen();
          return;
        }
        checkPlayPauseTouch(tx, ty);
        checkNextPreviousTouch(tx, ty);
        checkVolumeTouch(tx, ty);
        break;
      case PLAYLIST:
        handlePlaylistTouch(tx, ty);
        break;
      case SETTINGS:
        handleSettingsTouch(tx, ty);
        break;
    }
  }
}

// ----------------- Playlist -----------------
void drawPlaylistScreen() {
  tft.fillScreen(ILI9341_BLACK);
  tft.setTextSize(2);
  for(int i=0;i<totalTracks;i++){
    int y=40+i*40;
    uint16_t bg=(i%2==0)?ILI9341_DARKGREY:ILI9341_NAVY;
    tft.fillRect(0,y,SCR_W,40,bg);
    tft.setCursor(10,y+10);
    tft.setTextColor(ILI9341_WHITE);
    tft.print(trackList[i]);
  }
  // small hint to go to player
  tft.setTextSize(1);
  tft.setCursor(10, 8);
  tft.setTextColor(ILI9341_WHITE);
  tft.print("Tap a track to play");
}

void handlePlaylistTouch(int tx, int ty){
  for(int i=0; i<totalTracks; i++){
    int yStart = 40 + i*40;
    int yEnd = yStart + 40;
    if(ty > yStart && ty < yEnd){
      currentTrack = i;
      elapsedTime = 0;
      isPlaying = true;
      myDFPlayer.play(currentTrack * 2 + 1); // odd-track mapping
      goToPlayerScreen();
      return;
    }
  }
}

// ----------------- Player UI -----------------
void drawPlayerScreen(){
  // Draw static UI only once when entering player screen
  tft.fillScreen(ILI9341_BLACK);

  // Title bar background and static title (no scrolling)
  tft.fillRect(0, 0, SCR_W, titleBarH, ILI9341_DARKGREY);
  tft.setTextSize(2);
  tft.setTextColor(ILI9341_WHITE);
  tft.setCursor(4, 4);
  tft.print(trackList[currentTrack]);

  // Playlist button (top-left)
  tft.fillRect(playlistBtnX, playlistBtnY, playlistBtnW, playlistBtnH, ILI9341_NAVY);
  tft.setTextSize(2);
  tft.setCursor(playlistBtnX+2, playlistBtnY+1);
  tft.setTextColor(ILI9341_WHITE);
  tft.print(":"); // menu icon

  // Progress bar background (draw once)
  int px = 10, py = titleBarH + 6;
  int fullW = SCR_W - 20;
  tft.fillRect(px, py, fullW, 8, ILI9341_DARKGREY);
  prevBarWidth = 0;

  // Static time text (not updating per your selection)
  tft.setTextSize(1);
  tft.setTextColor(ILI9341_WHITE);
  tft.setCursor(10, titleBarH + 16);

  // Equalizer baseline: clear area once
  int baseY = (eqAreaTop + eqAreaBottom) / 2 + 10;
  int barWidth = 12, spacing = 14;
  int startX = (SCR_W - (5*barWidth + 4*spacing)) / 2;
  tft.fillRect(startX, baseY - 80, 5*(barWidth+spacing), 80, ILI9341_BLACK);
  for(int i=0;i<5;i++) prevEQHeights[i] = 0;

  // Draw initial pulsing circle background once
  int centerX = SCR_W/2;
  int centerY = 140;
  tft.fillCircle(centerX, centerY, 34, ILI9341_BLACK);

  // Draw nav buttons, play/pause and volume once
  drawPrevButton();
  drawPlayPauseButton();
  drawNextButton();
  drawVolumeSlider();
}

void goToPlayerScreen(){
  currentScreen = PLAYER;
  drawPlayerScreen();
}

// ----------------- Progress bar partial update -----------------
void updateProgressBar(){
  int px = 10, py = titleBarH + 6;
  int fullW = SCR_W - 20;
  int barWidth = constrain(map(elapsedTime, 0, trackDuration, 0, fullW), 0, fullW);

  // If bar grew, draw only the newly filled area
  if (barWidth > prevBarWidth) {
    uint16_t color = (elapsedTime < trackDuration*0.6) ? ILI9341_GREEN : ((elapsedTime < trackDuration*0.9) ? ILI9341_ORANGE : ILI9341_RED);
    tft.fillRect(px + prevBarWidth, py, barWidth - prevBarWidth, 8, color);
  }
  // If bar shrank (rare), erase the tail
  else if (barWidth < prevBarWidth) {
    tft.fillRect(px + barWidth, py, prevBarWidth - barWidth, 8, ILI9341_DARKGREY);
    uint16_t color = (elapsedTime < trackDuration*0.6) ? ILI9341_GREEN : ((elapsedTime < trackDuration*0.9) ? ILI9341_ORANGE : ILI9341_RED);
    tft.fillRect(px, py, barWidth, 8, color);
  }
  prevBarWidth = barWidth;
}

// ----------------- Equalizer partial update -----------------
void updateEqualizer(){
  int baseY = (eqAreaTop + eqAreaBottom) / 2 + 10;
  int barWidth = 12, spacing = 14;
  int startX = (SCR_W - (5*barWidth + 4*spacing)) / 2;

  for(int i=0;i<5;i++){
    int x = startX + i*(barWidth + spacing);
    int newHeight = random(12, 80);

    // If new is taller, draw the extra portion
    if(newHeight > prevEQHeights[i]){
      tft.fillRect(x, baseY - newHeight, barWidth, newHeight - prevEQHeights[i], ILI9341_GREEN);
    }
    // If new is shorter, clear the difference
    else if(newHeight < prevEQHeights[i]){
      tft.fillRect(x, baseY - prevEQHeights[i], barWidth, prevEQHeights[i] - newHeight, ILI9341_BLACK);
    }
    // Draw the current visible part (ensure top portion is green)
    tft.fillRect(x, baseY - newHeight, barWidth, newHeight, ILI9341_GREEN);

    prevEQHeights[i] = newHeight;
  }
}

// ----------------- Draw helpers (buttons / static parts) -----------------
void drawPlayPauseButton(){
  int x = playX, y = btnY, size = btnSize;
  // clear area around button (small area only)
  tft.fillRect(x-4, y-4, size+8, size+8, ILI9341_BLACK);
  // draw button bg
  tft.fillRect(x, y, size, size, ILI9341_DARKGREY);

  tft.setTextColor(ILI9341_WHITE);
  if(isPlaying){
    tft.fillRect(x+12, y+10, 8, 28, ILI9341_WHITE);
    tft.fillRect(x+28, y+10, 8, 28, ILI9341_WHITE);
  } else {
    tft.fillTriangle(x+14, y+10, x+14, y+38, x+34, y+24, ILI9341_WHITE);
  }
}

void drawNextButton() {
    uint16_t bg = tft.color565(50, 120, 200);
    uint16_t fg = ILI9341_WHITE;
    int btnW = SCR_W * 0.25;
    int btnH = SCR_H * 0.14;
    int x = SCR_W - btnW - 10;
    int y = SCR_H - btnH - 75;
    nextX = x;
    int cx = x + btnW / 2;
    tft.fillRoundRect(x, y, btnW, btnH, 10, bg);
    tft.fillTriangle(cx - 8, y + 8, cx - 8, y + btnH - 8, cx + 4, y + btnH / 2, fg);
    tft.fillTriangle(cx + 2, y + 8, cx + 2, y + btnH - 8, cx + 14, y + btnH / 2, fg);
    tft.fillRect(cx + 18, y + 8, 4, btnH - 16, fg);
}

void drawPrevButton() {
    uint16_t bg = tft.color565(50, 120, 200);
    uint16_t fg = ILI9341_WHITE;
    int btnW = SCR_W * 0.25;
    int btnH = SCR_H * 0.14;
    int x = 10;
    int y = SCR_H - btnH - 75;
    prevX = x;
    int cx = x + btnW / 2;
    tft.fillRoundRect(x, y, btnW, btnH, 10, bg);
    tft.fillRect(cx - 22, y + 8, 4, btnH - 16, fg);
    tft.fillTriangle(cx + 8, y + 8, cx + 8, y + btnH - 8, cx - 4, y + btnH / 2, fg);
    tft.fillTriangle(cx - 2, y + 8, cx - 2, y + btnH - 8, cx - 14, y + btnH / 2, fg);
}

void drawVolumeSlider(){
  // background
  tft.fillRect(volX, volY-2, volW, volH+4, ILI9341_BLACK);
  tft.fillRect(volX, volY, volW, volH, ILI9341_DARKGREY);
  int filled = map(volumeLevel, 0, 30, 0, volW);
  tft.fillRect(volX, volY, filled, volH, ILI9341_GREEN);
  tft.fillCircle(volX + filled, volY + volH/2, 6, ILI9341_WHITE);
}

// ----------------- Touch handlers -----------------
void checkPlayPauseTouch(int tx,int ty){
  int x = playX, y = btnY, size = btnSize;
  if(tx > x && tx < x+size && ty > y && ty < y+size){
    isPlaying = !isPlaying;
    if(isPlaying){
      myDFPlayer.play(currentTrack * 2 + 1);
    } else {
      myDFPlayer.pause();
    }
    drawPlayPauseButton();  // redraw ONLY when touched
  }
}


void checkNextPreviousTouch(int tx,int ty){
  int btnW = SCR_W * 0.25;
  int btnH = SCR_H * 0.14;
  int y = SCR_H - btnH - 75;

  // Next
  if(tx > nextX && tx < nextX + btnW && ty > y && ty < y + btnH){
    nextTrack();
    drawNextButton();     // redraw only when touched
    drawPlayPauseButton(); // track changed -> update
  }

  // Previous
  if(tx > prevX && tx < prevX + btnW && ty > y && ty < y + btnH){
    prevTrack();
    drawPrevButton();     // redraw only when touched
    drawPlayPauseButton();
  }
}


void checkVolumeTouch(int tx,int ty){
  if(tx > volX && tx < volX + volW && ty > volY-6 && ty < volY + volH + 6){
    int newVolume = map(tx - volX, 0, volW, 0, 30);
    newVolume = constrain(newVolume, 0, 30);
    if(newVolume != volumeLevel){
      volumeLevel = newVolume;
      myDFPlayer.volume(volumeLevel);
      drawVolumeSlider();
    }
  }
}

// ----------------- Track control helpers -----------------
void nextTrack(){
  currentTrack++;
  if(currentTrack >= totalTracks) currentTrack = 0;
  elapsedTime = 0;
  isPlaying = true;
  myDFPlayer.play(currentTrack * 2 + 1);
  drawPlayerScreen(); // redraw static UI for new track title/time baseline
}

void prevTrack(){
  currentTrack--;
  if(currentTrack < 0) currentTrack = totalTracks - 1;
  elapsedTime = 0;
  isPlaying = true;
  myDFPlayer.play(currentTrack * 2 + 1);
  drawPlayerScreen();
}

// ----------------- Settings (placeholder) -----------------
void drawSettingsScreen(){
  tft.fillScreen(ILI9341_BLACK);
  tft.setTextColor(ILI9341_WHITE);
  tft.setTextSize(2);
  tft.setCursor(50,30); tft.print("Settings");
}

void handleSettingsTouch(int tx,int ty){
  // implement touch logic for theme, brightness, EQ
}
