//*****************************************************************************
//
// blinky.c - Simple example to blink the on-board LED.
//
// Copyright (c) 2012-2020 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.2.0.295 of the EK-TM4C123GXL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/sysctl.h"
#include "driverlib/i2c.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"

#include "driverlib/interrupt.h"
#include "basic_conf.h"

//*********************************************************
// Function: I2C0_Init
// Description: Initializes I2C0 module on TM4C123G for Master mode
//*********************************************************
void I2C0_Init(void)
{
    // Enable I2C0 and GPIOB peripherals
    SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);

    // Wait until peripherals are ready
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_I2C0));
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB));

    // Configure PB2 as I2C SCL and PB3 as I2C SDA
    GPIOPinConfigure(GPIO_PB2_I2C0SCL);
    GPIOPinConfigure(GPIO_PB3_I2C0SDA);

    GPIOPinTypeI2CSCL(GPIO_PORTB_BASE, GPIO_PIN_2);  // Configure PB2 -> SCL
    GPIOPinTypeI2C(GPIO_PORTB_BASE, GPIO_PIN_3);  // Configure PB3 -> SDA

    // Initialize I2C master with system clock
    // false = standard mode (100kHz), true = fast mode (400kHz)
    I2CMasterInitExpClk(I2C0_BASE, SysCtlClockGet(), false);

    // Clear any previous I2C status
    I2CMasterIntClear(I2C0_BASE);
}

void I2C_WriteByte(uint8_t slaveAddr, uint16_t memAddr, uint8_t data)
{
    // Send slave address with WRITE bit (R/W = 0)
    I2CMasterSlaveAddrSet(I2C0_BASE, slaveAddr, false);

    // Send high byte of memory address
    I2CMasterDataPut(I2C0_BASE, (memAddr >> 8) & 0xFF);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_START);  // Start new Write
    while(I2CMasterBusy(I2C0_BASE));

    // Send low byte of memory address
    I2CMasterDataPut(I2C0_BASE, memAddr & 0xFF);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_CONT);  // Continue Write
    while(I2CMasterBusy(I2C0_BASE));

    // Send the actual data byte
    I2CMasterDataPut(I2C0_BASE, data);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);  // Finish Write
    while(I2CMasterBusy(I2C0_BASE));
}

uint8_t I2C_ReadByte(uint8_t slaveAddr, uint16_t memAddr)
{
    uint8_t data;

    // Send slave address with WRITE bit (R/W = 0)
    I2CMasterSlaveAddrSet(I2C0_BASE, slaveAddr, false);

    // Send high byte of memory address
    I2CMasterDataPut(I2C0_BASE, (memAddr >> 8) & 0xFF);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_START);  // Start new Write
    while(I2CMasterBusy(I2C0_BASE));

    // Send low byte of memory address
    I2CMasterDataPut(I2C0_BASE, memAddr & 0xFF);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);  // Finish Write
    while(I2CMasterBusy(I2C0_BASE));

    // Repeated START + Slave address with READ bit (R/W = 1)
    I2CMasterSlaveAddrSet(I2C0_BASE, slaveAddr, true);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);  // Read 1 Byte
    while(I2CMasterBusy(I2C0_BASE));

    // Read the received byte
    data = I2CMasterDataGet(I2C0_BASE);
    return data;
}

uint8_t RxData = 0;

int main(void)
{
    systemClockConfig();
    I2C0_Init();
    IntMasterEnable();


    // Write and Read Test
    I2C_WriteByte(0x50, 0x0010, 0xAC);    // Write 0xAB to address 0x0010
    delay_ms(5);    // Small delay for write cycle
    RxData = I2C_ReadByte(0x50, 0x0010);

    while(1){
        delay_ms(1);
    }
}
