/*
 * TCP_Client.c
 *
 *  Created on: Nov 10, 2025
 *      Author: controllerstech
 */


#include "TCP_Client.h"
#include "wizchip_conf.h"
#include "socket.h"
#include "dns/dns.h"
#include "stdio.h"
#include "string.h"
#include "main.h"



#define DNS_RESOLVE			   1  // comment out to use static SERVER IP


// Configuration
#define TCP_CLIENT_SOCKET      0
#define TCP_SERVER_PORT        4242
#define TCP_SERVER_HOST        "tcpbin.com"   // <-- hostname : 4242
#define DNS_SOCKET             6                       // separate socket for DNS
#define DNS_SERVER_IP          {8, 8, 8, 8}            // Google's DNS

#if !defined(DNS_RESOLVE)
#define TCP_SERVER_IP  {192, 168, 1, 7}
#endif


uint8_t tx_buf[256];
uint8_t rx_buf[256];
uint8_t dns_buffer[512];

static uint8_t server_ip[4];
static uint8_t dns_ip[4] = DNS_SERVER_IP;

/**
 * @brief Resolve hostname to IP using DNS
 * @retval 0 on success, -1 on failure
 */
int TCP_Client_ResolveDNS(void)
{
	printf("Resolving %s ...\r\n", TCP_SERVER_HOST);

	DNS_init(DNS_SOCKET, dns_buffer);

	int8_t result = DNS_run(dns_ip, (uint8_t *)TCP_SERVER_HOST, server_ip);

	if (result != 1)
	{
		printf("DNS resolution failed (code=%d)\r\n", result);
		return -1;
	}

	printf("Resolved %s -> %d.%d.%d.%d\r\n", TCP_SERVER_HOST,
			server_ip[0], server_ip[1], server_ip[2], server_ip[3]);
	return 0;
}


/**
 * @brief Initialize TCP client and connect to server
 * @retval 0 on success, -1 on failure
 */
int TCP_Client_Init(void)
{
	int8_t ret;
	uint8_t status;

	printf("Initializing TCP Client...\r\n");

#ifdef DNS_RESOLVE
	// Step 1: Resolve the hostname
	if (TCP_Client_ResolveDNS() != 0)
		return -1;
#else
    uint8_t static_ip[4] = TCP_SERVER_IP;
    memcpy(server_ip, static_ip, 4);
#endif

	// Step 2: Create socket
	ret = socket(TCP_CLIENT_SOCKET, Sn_MR_TCP, 3000, 0);
	if (ret != TCP_CLIENT_SOCKET)
	{
		printf("Socket open failed\r\n");
		return -1;
	}

	// Step 3: Connect to remote server
	printf("Connecting to (%d.%d.%d.%d:%d)...\r\n",
			server_ip[0], server_ip[1], server_ip[2], server_ip[3], TCP_SERVER_PORT);

	ret = connect(TCP_CLIENT_SOCKET, server_ip, TCP_SERVER_PORT);
	if (ret != SOCK_OK)
	{
		printf("Connect() failed (ret=%d)\r\n", ret);
		close(TCP_CLIENT_SOCKET);
		return -1;
	}

	// Step 4: Wait until the socket is established
	uint32_t timeout = HAL_GetTick() + 5000;
	do {
		status = getSn_SR(TCP_CLIENT_SOCKET);
		if (status == SOCK_ESTABLISHED)
		{
			printf("TCP connection established!\r\n");
			return 0;
		}
		if (status == SOCK_CLOSED || status == SOCK_CLOSE_WAIT)
			break;
	} while (HAL_GetTick() < timeout);

	printf("Connection timeout or closed\r\n");
	close(TCP_CLIENT_SOCKET);
	return -1;
}


/**
 * @brief Send and receive data through the established TCP connection
 * @param data: pointer to data string to send
 */
void TCP_Client_Handle(const char *data)
{
	int32_t ret;
	uint8_t status;

	status = getSn_SR(TCP_CLIENT_SOCKET);

	if (status == SOCK_ESTABLISHED)
	{
		// Step 1: Send data
		if (data && strlen(data) > 0)
		{
			ret = send(TCP_CLIENT_SOCKET, (uint8_t *)data, strlen(data));
			if (ret > 0)
				printf("Sent (%ld bytes): %s\r\n", ret, data);
			else
				printf("Send failed (ret=%ld)\r\n", ret);
		}

		// Step 2: Receive data
		ret = recv(TCP_CLIENT_SOCKET, rx_buf, sizeof(rx_buf) - 1);
		if (ret > 0)
		{
			rx_buf[ret] = '\0';
			printf("Received (%ld bytes): %s\r\n", ret, rx_buf);
		}
		else if (ret < 0)
		{
			printf("Receive error (ret=%ld)\r\n", ret);
		}
	}
	else if (status == SOCK_CLOSE_WAIT)
	{
		printf("Server closed connection.\r\n");
		disconnect(TCP_CLIENT_SOCKET);
	}
	else if (status == SOCK_CLOSED)
	{
		printf("Socket closed, attempting reconnect...\r\n");
		TCP_Client_Init();
	}
}
