#include "lcd1602_i2c.h"
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include "inc/hw_memmap.h"
#include "driverlib/i2c.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "basic_conf.h"

uint8_t lcd_backlight_state = LCD_BACKLIGHT;

// ------------------------------------------------------
//  LOW-LEVEL I2C WRITE FUNCTION
// ------------------------------------------------------
static void LCD_I2C_Write(uint8_t data)
{
    I2CMasterSlaveAddrSet(I2C0_BASE, LCD_I2C_ADDR, false);
    I2CMasterDataPut(I2C0_BASE, data);
    I2CMasterControl(I2C0_BASE, I2C_MASTER_CMD_SINGLE_SEND);

    while(I2CMasterBusy(I2C0_BASE));
}

// ------------------------------------------------------
//  SEND 4-BIT ENABLE PULSE
// ------------------------------------------------------
static void LCD_SendNibble(uint8_t nibble)
{
    LCD_I2C_Write(nibble | lcd_backlight_state | LCD_ENABLE_BIT);
    delay_ms(1);

    LCD_I2C_Write(nibble | lcd_backlight_state);
    delay_ms(1);
}

// ------------------------------------------------------
//  SEND COMMAND BYTE (HIGH NIBBLE + LOW NIBBLE)
// ------------------------------------------------------
void LCD_SendCommand(uint8_t cmd)
{
    uint8_t highNib = cmd & 0xF0;  // RS = 0 for command
    uint8_t lowNib  = (cmd << 4) & 0xF0;  // RS = 0 for command

    LCD_SendNibble(highNib);
    LCD_SendNibble(lowNib);
}

// ------------------------------------------------------
//  SEND DATA BYTE (CHARACTER)
// ------------------------------------------------------
void LCD_SendData(uint8_t data)
{
    uint8_t highNib = (data & 0xF0) | 0x01;  // RS = 1 for Data
    uint8_t lowNib  = ((data << 4) & 0xF0) | 0x01;  // RS = 1 for Data

    LCD_SendNibble(highNib);
    LCD_SendNibble(lowNib);
}

// ------------------------------------------------------
//  CLEAR LCD
// ------------------------------------------------------
void LCD_Clear(void)
{
    LCD_SendCommand(LCD_CLEAR);
    delay_ms(2);
}

// ------------------------------------------------------
//  BACKLIGHT CONTROL
// ------------------------------------------------------
void LCD_BacklightOn(void)
{
    lcd_backlight_state = LCD_BACKLIGHT;
    LCD_I2C_Write(lcd_backlight_state);
}

void LCD_BacklightOff(void)
{
    lcd_backlight_state = LCD_NOBACKLIGHT;
    LCD_I2C_Write(lcd_backlight_state);
}

// ------------------------------------------------------
//  SET CURSOR POSITION
// ------------------------------------------------------
void LCD_SetCursor(uint8_t row, uint8_t col)
{
    uint8_t address = 0x00;

    switch (row)
    {
        case 0: address = 0x00 + col; break;
        case 1: address = 0x40 + col; break;
    }

    LCD_SendCommand(LCD_SET_CURSOR | address);
}

// ------------------------------------------------------
//  PRINT STRING
// ------------------------------------------------------
void LCD_PrintString(char *str)
{
    while(*str)
    {
        LCD_SendData(*str++);
    }
}

// ------------------------------------------------------
//  PRINT INTEGER
// ------------------------------------------------------
void LCD_PrintInt(int num)
{
    char buffer[12];
    sprintf(buffer, "%d", num);
    LCD_PrintString(buffer);
}

// ------------------------------------------------------
//  PRINT FLOAT
// ------------------------------------------------------
static void float_to_string(float num, char *buffer, int precision);
void LCD_PrintFloat(float num, int decimalPlaces)
{
    char buffer[20];
    float_to_string(num, buffer, decimalPlaces);
    LCD_PrintString(buffer);
}

// ------------------------------------------------------
//  CREATE CUSTOM CHARACTER (8 BYTES)
// ------------------------------------------------------
void LCD_CreateChar(uint8_t location, uint8_t charmap[])
{
    location &= 0x07;
    LCD_SendCommand(0x40 | (location << 3));

    int i=0;
    for (i= 0; i < 8; i++)
    {
        LCD_SendData(charmap[i]);
    }
}

// ------------------------------------------------------
//  SCROLL DISPLAY LEFT BY 1 PLACE
// ------------------------------------------------------
void LCD_ScrollLeft(void)
{
    // 0x18 = Shift entire display left
    LCD_SendCommand(0x18);
}

// ------------------------------------------------------
//  SCROLL DISPLAY RIGHT BY 1 PLACE
// ------------------------------------------------------
void LCD_ScrollRight(void)
{
    // 0x1C = Shift entire display right
    LCD_SendCommand(0x1C);
}

// ------------------------------------------------------
//  LCD INITIALIZATION SEQUENCE
// ------------------------------------------------------
void LCD_Init(void)
{
    delay_ms(50);  // wait > 40ms

    LCD_SendNibble(0x30);
    delay_ms(5);

    LCD_SendNibble(0x30);
    delay_ms(5);

    LCD_SendNibble(0x30);
    delay_ms(1);

    LCD_SendNibble(0x20);

    LCD_SendCommand(LCD_FUNCTION_SET);
    LCD_SendCommand(LCD_DISPLAY_ON);
    LCD_SendCommand(LCD_CLEAR);
    LCD_SendCommand(LCD_ENTRY_MODE);
}


void float_to_string(float num, char *buffer, int precision)
{
    // Handle negative numbers
    if (num < 0) {
        *buffer++ = '-';
        num = -num;
    }

    // Apply rounding:
    // Example: for precision=3 → add 0.0005 before truncation
    float rounding = 0.5f;
    int i = 0;
    for (i = 0; i < precision; i++)
        rounding /= 10.0f;

    num += rounding;

    // Extract integer part
    int int_part = (int)num;

    // Extract fractional part
    float frac = num - (float)int_part;

    // Convert integer part
    char temp[16];
    i = 0;
    if (int_part == 0) {
        temp[i++] = '0';
    } else {
        while (int_part > 0) {
            temp[i++] = (int_part % 10) + '0';
            int_part /= 10;
        }
    }

    // Reverse digits
    while (i--)
        *buffer++ = temp[i];

    // If no decimals requested
    if (precision == 0) {
        *buffer = '\0';
        return;
    }

    // Decimal point
    *buffer++ = '.';

    // Fractional digits
    int p=0;
    for (p = 0; p < precision; p++) {
        frac *= 10.0f;
        int digit = (int)frac;
        *buffer++ = digit + '0';
        frac -= digit;
    }

    // End string
    *buffer = '\0';
}
