//*****************************************************************************
//
// blinky.c - Simple example to blink the on-board LED.
//
// Copyright (c) 2012-2020 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.2.0.295 of the EK-TM4C123GXL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/uart.h"
#include "driverlib/pin_map.h"
#include "driverlib/interrupt.h"
#include "basic_conf.h"
#include "inc/hw_ints.h"
#include "string.h"


#define RX_BUFFER_SIZE 64
char rxBuffer[RX_BUFFER_SIZE];
volatile uint8_t isRxed = 0;
volatile uint8_t indx = 0;
void UART0_Handler(void);

// Initialize UART0 with RX interrupt
void UART0_InterruptInit(void)
{
    // Enable UART0 and Port A
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    // Wait for peripherals to be ready
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_UART0));
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOA));

    // Configure pins for UART0 (PA0=RX, PA1=TX)
    GPIOPinConfigure(GPIO_PA0_U0RX);
    GPIOPinConfigure(GPIO_PA1_U0TX);
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    // Configure UART clock and baud rate
    UARTClockSourceSet(UART0_BASE, UART_CLOCK_PIOSC);
    UARTConfigSetExpClk(UART0_BASE, 16000000, 115200,
                        (UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));

    // Register UART0 Interrupt Handler
    UARTIntRegister(UART0_BASE, UART0_Handler);

    // Enable UART receive interrupt
    UARTIntEnable(UART0_BASE, UART_INT_RX | UART_INT_RT);    // RX interrupt only

    IntEnable(INT_UART0);

    // Enable global interrupts
    // IntMasterEnable();  // we will do this later in the main function

    // Enable UART
    UARTEnable(UART0_BASE);
}

void UART0_SendString(const char *str)
{
    while(*str)
    {
        UARTCharPut(UART0_BASE, *str++);
    }
}

int main(void)
{
    systemClockConfig();
    UART0_InterruptInit();

    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOF));

    GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_3|GPIO_PIN_1|GPIO_PIN_2);

    IntMasterEnable();

    while(1)
    {
        if (isRxed == 1)
        {
            isRxed = 0;
            if (strncmp(rxBuffer, "RED", indx) == 0)
                GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3, GPIO_PIN_1);
            else if (strncmp(rxBuffer, "BLUE", indx) == 0)
                GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3, GPIO_PIN_2);
            else if (strncmp(rxBuffer, "GREEN", indx) == 0)
                GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3, GPIO_PIN_3);
            else if (strncmp(rxBuffer, "PINK", indx) == 0)
                GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3, GPIO_PIN_1|GPIO_PIN_2);
            else if (strncmp(rxBuffer, "YELLOW", indx) == 0)
                GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3, GPIO_PIN_1|GPIO_PIN_3);
            else
                GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1 | GPIO_PIN_2 | GPIO_PIN_3, 0);
        }

        UART0_SendString("Hello World\r\n");
        delay_ms(1000);
    }
}

void UART0_Handler(void)
{
    uint32_t status = UARTIntStatus(UART0_BASE, true);
    UARTIntClear(UART0_BASE, status);
    if ((status & UART_INT_RX) || (status & UART_INT_RT))  // If the RX interrupt is triggered
    {
        indx = 0;
        while(UARTCharsAvail(UART0_BASE))  // check if there are more available characters
        {
            char data = UARTCharGetNonBlocking(UART0_BASE);  // Read data in Non-Blocking manner
            rxBuffer[indx++] = data;  // store data in the buffer
//            if(indx >= RX_BUFFER_SIZE) indx = 0;  // Wrap around for circular buffer
            isRxed = 1;
        }
    }
}
