//*****************************************************************************
//
// blinky.c - Simple example to blink the on-board LED.
//
// Copyright (c) 2012-2020 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.2.0.295 of the EK-TM4C123GXL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "utils/uartstdio.h"
#include "driverlib/uart.h"
#include "driverlib/interrupt.h"
#include "basic_conf.h"

#include "stdio.h"

void UART0_Init(void)
{
    // Enable UART0 and Port A
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    // Wait for peripherals to be ready
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_UART0));
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOA));

    // Configure pins for UART0 (PA0=RX, PA1=TX)
    GPIOPinConfigure(GPIO_PA0_U0RX);
    GPIOPinConfigure(GPIO_PA1_U0TX);
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    // Configure UART clock and baud rate
    UARTClockSourceSet(UART0_BASE, UART_CLOCK_PIOSC);
    UARTConfigSetExpClk(UART0_BASE, 16000000, 115200,
                        (UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));

    // Enable UART
    UARTEnable(UART0_BASE);
}

void UART0_SendString(const char *str)
{
    while(*str)
    {
        UARTCharPut(UART0_BASE, *str++);
    }
}

void UART0_SendInt(int num)
{
    char buffer[12];
    sprintf(buffer, "%d", num);    // convert integer to string
    UART0_SendString(buffer);
    UART0_SendString("\r\n");
}

void UART0_SendFloat(float num)
{
    int int_part = (int)num;
    int frac_part = (int)((num - int_part) * 100);
    char buffer[20];
    sprintf(buffer, "%d.%02d", int_part, frac_part);
    UART0_SendString(buffer);
}

main(void)
{
    systemClockConfig();
    UART0_Init();
    IntMasterEnable();

    // UART0_SendString("Hello from TM4C123G!\r\n");
    // UART0_SendInt(1234);
    // UART0_SendFloat(25.67);
    
    while(1)
    {
        if(UARTCharsAvail(UART0_BASE))                      // If char is available
        {
            char c = UARTCharGetNonBlocking(UART0_BASE);   // read from uart
            UARTCharPut(UART0_BASE, c);            // echo back
        }
    }
}
