#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SH110X.h>

#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
Adafruit_SH1106G display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

void drawPacman(int x, int y, bool mouthOpen) {
  int r = 12;  // Even bigger radius: 24x24 Pacman
  int cx = x + r;
  int cy = y + r;
  display.fillCircle(cx, cy, r, SH110X_WHITE);
  if (mouthOpen) {
    // Black triangle for open mouth facing right (scaled up further)
    display.fillTriangle(cx, cy, cx + r, cy - 6, cx + r, cy + 6, SH110X_BLACK);
  }
}

void drawDot(int x, int y) {
  // Slightly bigger dot: 6x6 filled square for visibility with huge Pacman
  display.fillRect(x, y, 6, 6, SH110X_WHITE);
}

void setup() {
  display.begin(0x3C, true);  // Use 0x3C for most SH1106G OLEDs; adjust if needed
  
  // Clear the entire screen during initialization
  display.clearDisplay();
  display.display();  // Force immediate clear to black
  
  // Initial big dots (6x6 squares, spaced wider for 24x24 Pacman)
  int dotY = 30;  // Moved up to center vertically now that WiFi is gone
  int dotPositions[5] = {12, 35, 58, 81, 104};
  for (int i = 0; i < 5; i++) {
    drawDot(dotPositions[i], dotY);
  }
  
  // Update display after all initial draws
  display.display();
}

void loop() {
  // Pacman animation: moves right, eats dots (erases them), chomps mouth, resets on screen cross
  static int pacX = -24;  // Start off-screen left (bigger size)
  static bool mouthOpen = true;
  static bool eaten[5] = {false};  // Match numDots
  static unsigned long lastUpdate = 0;
  const int pacY = 18;  // Pacman top-left Y (centered vertically)
  const int dotY = 30;  // Dots centered with Pacman
  int dotPositions[5] = {12, 35, 58, 81, 104};
  const int numDots = 5;
  const int moveSpeed = 4;  // Faster to traverse screen quicker with massive size
  const int animationDelay = 120;  // ms delay (balanced for big visual heft)

  unsigned long currentTime = millis();
  if (currentTime - lastUpdate >= animationDelay) {
    // Clear old Pacman position if on screen (bigger clear area)
    if (pacX >= 0 && pacX < SCREEN_WIDTH) {
      display.fillRect(pacX, pacY, 24, 24, SH110X_BLACK);  // Clear 24x24 area
    }

    // Move Pacman
    pacX += moveSpeed;
    bool shouldReset = (pacX > SCREEN_WIDTH);

    if (shouldReset) {
      // Reset animation: reposition Pacman and restore all dots
      pacX = -24;
      for (int i = 0; i < numDots; i++) {
        eaten[i] = false;
        drawDot(dotPositions[i], dotY);
      }
    } else {
      // Check for eating dots at new position (wider range for huge Pacman)
      for (int i = 0; i < numDots; i++) {
        if (!eaten[i] && abs((pacX + 12) - (dotPositions[i] + 3)) <= 12) {  // Center-to-center <=12
          eaten[i] = true;
          display.fillRect(dotPositions[i], dotY, 6, 6, SH110X_BLACK);  // Erase dot
        }
      }
    }

    // Draw new Pacman if on screen
    if (pacX >= 0) {
      drawPacman(pacX, pacY, mouthOpen);
      mouthOpen = !mouthOpen;  // Toggle for chomping
    }

    // Update display
    display.display();
    lastUpdate = currentTime;
  }
}