#include <LedControl.h>
#include "fonts.h"

// Pin connections
const int DIN_PIN = 11;   // Data (MOSI)
const int CLK_PIN = 13;   // Clock (SCK)
const int CS_PIN  = 10;   // Chip Select (LOAD)

// Number of cascaded MAX7219 modules
const int NUM_MODULES = 1;

#define ENABLE_SEGMENT_REMAP   1   // Set to 1 if characters look mirrored
#define REVERSE_DIGIT_ORDER    0   // Set to 1 if digits appear reversed

LedControl lc = LedControl(DIN_PIN, CLK_PIN, CS_PIN, NUM_MODULES);

// Counter speed control
const int counterDelay = 200;  // Delay between updates (ms)

void setup() {
  lc.shutdown(0, false);   // Wake up MAX7219
  lc.setIntensity(0, 7);   // Set brightness (0–15)
  lc.clearDisplay(0);      // Clear all digits
}

void loop() {
  displayCounter();
}

//---------------------------------------------------------------
// Helper function to fix segment bit order if needed
//---------------------------------------------------------------
uint8_t remap7seg(uint8_t v) {
  uint8_t dp = v & 0x80;        // Keep decimal point (bit 7)
  uint8_t low7 = v & 0x7F;      // Extract lower 7 bits (A–G)
  uint8_t rev = 0;
  for (uint8_t i = 0; i < 7; ++i)
    if (low7 & (1 << i)) rev |= (1 << (6 - i));
  return dp | rev;
}

//---------------------------------------------------------------
// Function to increment and display numbers continuously
//---------------------------------------------------------------
void displayCounter() {
  for (long num = 0; num <= 99999999; num++) {
    showNumber(num);
    delay(counterDelay);
  }
}

//---------------------------------------------------------------
// Function to display an 8-digit number
//---------------------------------------------------------------
void showNumber(long value) {
  char buffer[9];
  sprintf(buffer, "%8ld", value);  // Format to 8 characters (right-aligned)

  for (int pos = 0; pos < 8; pos++) {
    char c = buffer[pos];

#if REVERSE_DIGIT_ORDER
    int displayPos = pos;
#else
    int displayPos = 7 - pos;
#endif

    byte pattern = FONT_7SEG[(int)c];

#if ENABLE_SEGMENT_REMAP
    pattern = remap7seg(pattern);
#endif

    lc.setRow(0, displayPos, pattern);
  }
}
