// Full ADXL345 Tilt Angle Code (Pitch & Roll)
// Includes: initialization, calibration, raw values, g-values, pitch, roll

#include <Wire.h>

#define ADXL345_ADDR 0x53   // SDO = GND

const float SCALE_FACTOR = 256.0;   // LSB/g in full-resolution mode
const int CAL_SAMPLES = 200;
const unsigned long LOOP_DELAY_MS = 200;

float xOffset = 0.0, yOffset = 0.0, zOffset = 0.0;

// Write to ADXL345 register
void adxlWrite(uint8_t reg, uint8_t value) {
  Wire.beginTransmission(ADXL345_ADDR);
  Wire.write(reg);
  Wire.write(value);
  Wire.endTransmission();
}

// Setup registers
void initADXL345() {
  adxlWrite(0x2D, 0x08); // POWER_CTL: measurement mode
  adxlWrite(0x31, 0x08); // DATA_FORMAT: full resolution, ±2g
  adxlWrite(0x2C, 0x0A); // BW_RATE: 100 Hz
}

// Read 6 bytes from DATAX0
void readRaw(int16_t &x, int16_t &y, int16_t &z) {
  Wire.beginTransmission(ADXL345_ADDR);
  Wire.write(0x32);  // DATAX0
  Wire.endTransmission(false);

  Wire.requestFrom(ADXL345_ADDR, 6, true);
  x = (int16_t)(Wire.read() | (Wire.read() << 8));
  y = (int16_t)(Wire.read() | (Wire.read() << 8));
  z = (int16_t)(Wire.read() | (Wire.read() << 8));
}

// Average-based calibration
void calibrate() {
  Serial.println("Calibration: keep sensor flat and still...");

  long xSum = 0, ySum = 0, zSum = 0;

  for (int i = 0; i < CAL_SAMPLES; i++) {
    int16_t xr, yr, zr;
    readRaw(xr, yr, zr);

    xSum += xr;
    ySum += yr;
    zSum += zr;

    delay(5);
  }

  float xAvg = xSum / CAL_SAMPLES;
  float yAvg = ySum / CAL_SAMPLES;
  float zAvg = zSum / CAL_SAMPLES;

  // X & Y should be 0g, Z should be +1g
  xOffset = xAvg;
  yOffset = yAvg;
  zOffset = zAvg - SCALE_FACTOR;

  Serial.println("Offsets calculated:");
  Serial.print("X Offset: "); Serial.println(xOffset);
  Serial.print("Y Offset: "); Serial.println(yOffset);
  Serial.print("Z Offset: "); Serial.println(zOffset);
  Serial.println();
}

void setup() {
  Serial.begin(9600);
  Wire.begin();
  delay(50);

  initADXL345();
  delay(50);

  calibrate();

  Serial.println("Starting angle calculations...\n");
}

void loop() {
  int16_t xRaw, yRaw, zRaw;
  readRaw(xRaw, yRaw, zRaw);

  // Apply offsets
  float xCal = xRaw - xOffset;
  float yCal = yRaw - yOffset;
  float zCal = zRaw - zOffset;

  // Convert to g
  float Ax = xCal / SCALE_FACTOR;
  float Ay = yCal / SCALE_FACTOR;
  float Az = zCal / SCALE_FACTOR;

  // Tilt Angle Calculation (Option A)
  float pitch = atan2(Ax, sqrt(Ay * Ay + Az * Az)) * 180.0 / PI;
  float roll  = atan2(Ay, sqrt(Ax * Ax + Az * Az)) * 180.0 / PI;

  // Print output
  Serial.print("Pitch: "); Serial.print(pitch, 2); Serial.print("°  ");
  Serial.print("Roll: ");  Serial.print(roll, 2);  Serial.println("°");

  delay(LOOP_DELAY_MS);
}