// Full ADXL345 sketch:
// - Init (full resolution, 100Hz)
// - Auto calibration (200 samples)
// - Read raw values
// - Convert to g
// - Compute Pitch & Roll (Option A)
// - Print Raw, g, Pitch, Roll

#include <Wire.h>

#define ADXL345_ADDR 0x53   // SDO = GND

const float SCALE_FACTOR = 256.0;   // LSB per g in full-res ±2g mode
const int CAL_SAMPLES = 200;
const unsigned long LOOP_DELAY_MS = 200;

float xOffset = 0.0;
float yOffset = 0.0;
float zOffset = 0.0;

void adxlWriteRegister(uint8_t reg, uint8_t value) {
  Wire.beginTransmission(ADXL345_ADDR);
  Wire.write(reg);
  Wire.write(value);
  Wire.endTransmission();
}

void setupADXL345() {
  // Measurement mode
  adxlWriteRegister(0x2D, 0x08);   // POWER_CTL: Measure bit = 1

  // Full resolution, ±2g
  adxlWriteRegister(0x31, 0x08);   // DATA_FORMAT: Full res

  // Data rate = 100 Hz
  adxlWriteRegister(0x2C, 0x0A);   // BW_RATE: 0x0A => 100 Hz
}

void readRawXYZ(int16_t &x, int16_t &y, int16_t &z) {
  Wire.beginTransmission(ADXL345_ADDR);
  Wire.write(0x32);            // DATAX0
  Wire.endTransmission(false);

  Wire.requestFrom(ADXL345_ADDR, 6, true);
  x = (int16_t)(Wire.read() | (Wire.read() << 8));
  y = (int16_t)(Wire.read() | (Wire.read() << 8));
  z = (int16_t)(Wire.read() | (Wire.read() << 8));
}

void calibrateSensor() {
  Serial.println("Calibration: Keep sensor flat and still...");
  long xSum = 0, ySum = 0, zSum = 0;

  for (int i = 0; i < CAL_SAMPLES; i++) {
    int16_t xr, yr, zr;
    readRawXYZ(xr, yr, zr);
    xSum += xr;
    ySum += yr;
    zSum += zr;
    delay(5);
  }

  float xAvg = (float)xSum / CAL_SAMPLES;
  float yAvg = (float)ySum / CAL_SAMPLES;
  float zAvg = (float)zSum / CAL_SAMPLES;

  // Offsets to subtract from raw readings
  xOffset = xAvg;                // X should be ~0g when flat
  yOffset = yAvg;                // Y should be ~0g when flat
  zOffset = zAvg - SCALE_FACTOR; // Z should be ~1g (≈ SCALE_FACTOR) when flat

  Serial.println("Calibration complete.");
  Serial.print("X Offset: "); Serial.println(xOffset);
  Serial.print("Y Offset: "); Serial.println(yOffset);
  Serial.print("Z Offset: "); Serial.println(zOffset);
  Serial.println();
  delay(200);
}

void setup() {
  Serial.begin(9600);
  Wire.begin();
  delay(50);

  setupADXL345();
  delay(50);

  Serial.println("ADXL345 initialized (Full res, 100Hz).");
  calibrateSensor();
  Serial.println("Starting readings...");
  Serial.println();
}

void loop() {
  int16_t xRaw, yRaw, zRaw;
  readRawXYZ(xRaw, yRaw, zRaw);

  // Apply calibration offsets (raw)
  float xCal = (float)xRaw - xOffset;
  float yCal = (float)yRaw - yOffset;
  float zCal = (float)zRaw - zOffset;

  // Convert to g
  float xg = xCal / SCALE_FACTOR;
  float yg = yCal / SCALE_FACTOR;
  float zg = zCal / SCALE_FACTOR;

  // Compute Pitch & Roll (Option A)
  // Pitch = atan2( Xg , sqrt(Yg^2 + Zg^2) )
  // Roll  = atan2( Yg , sqrt(Xg^2 + Zg^2) )
  float pitch = atan2(xg, sqrt(yg * yg + zg * zg)) * 180.0 / PI;
  float roll  = atan2(yg, sqrt(xg * xg + zg * zg)) * 180.0 / PI;

  // Print Raw + g + angles
  Serial.print("RawX: "); Serial.print(xRaw);
  Serial.print(" RawY: "); Serial.print(yRaw);
  Serial.print(" RawZ: "); Serial.print(zRaw);

  Serial.print("  ||  X(g): "); Serial.print(xg, 3);
  Serial.print(" Y(g): "); Serial.print(yg, 3);
  Serial.print(" Z(g): "); Serial.print(zg, 3);

  Serial.print("  ||  Pitch: "); Serial.print(pitch, 2);
  Serial.print("°  Roll: "); Serial.print(roll, 2);
  Serial.println("°");

  delay(LOOP_DELAY_MS);
}