#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/sysctl.h"
#include "driverlib/systick.h"
#include "driverlib/interrupt.h"   // for IntMasterEnable


// System Clock Related Functions
void systemClockConfig (void)
{
    // Using MOSC 16MHz (External) -> PLL (400MHz) -> PLL (200MHz)/2.5 = 80MHz
    SysCtlClockSet(SYSCTL_XTAL_16MHZ|SYSCTL_USE_PLL|SYSCTL_OSC_MAIN|SYSCTL_SYSDIV_2_5);
}

// SysTick Timer Related functions
volatile uint32_t msTicks = 0;

void SysTick_Handler(void)
{
    msTicks++;   // increment every millisecond
}

void delay_ms(uint32_t ms)
{
    uint32_t start = msTicks;
    while ((msTicks - start) < ms) {}
}

void SysTick_Init(void)
{
    SysTickPeriodSet(SysCtlClockGet() / 1000); // 1ms period
    SysTickIntRegister(SysTick_Handler);     // Register ISR
    SysTickIntEnable();                        // Enable SysTick interrupt
    SysTickEnable();                           // Start SysTick
}

int main(void)
{
    // Set system clock
    systemClockConfig();

    IntMasterEnable();  // enable master interrupt
    SysTick_Init();  // configure SysTick Timer

    // Enable GPIOF
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOF));

    // Configure PF1 as output (Red LED)
    GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_1);

    while(1)
    {
        GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1, GPIO_PIN_1);  // LED ON
        delay_ms(1000);  // Simple delay
        GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1, 0);           // LED OFF
        delay_ms(1000);  // Simple delay
    }
}
