#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/sysctl.h"
#include "driverlib/systick.h"
#include "driverlib/interrupt.h"   // for IntMasterEnable


// System Clock Related Functions
void systemClockConfig (void)
{
    // Using MOSC 16MHz (External) -> PLL (400MHz) -> PLL (200MHz)/2.5 = 80MHz
    SysCtlClockSet(SYSCTL_XTAL_16MHZ|SYSCTL_USE_PLL|SYSCTL_OSC_MAIN|SYSCTL_SYSDIV_2_5);
}

// SysTick Timer Related functions
volatile uint32_t msTicks = 0;

void SysTick_Handler(void)
{
    msTicks++;   // increment every millisecond
}

void delay_ms(uint32_t ms)
{
    uint32_t start = msTicks;
    while ((msTicks - start) < ms) {}
}

void SysTick_Init(void)
{
    SysTickPeriodSet(SysCtlClockGet() / 1000); // 1ms period
    SysTickIntRegister(SysTick_Handler);     // Register ISR
    SysTickIntEnable();                        // Enable SysTick interrupt
    SysTickEnable();                           // Start SysTick
}

int main(void)
{
    // Set system clock
    systemClockConfig();

    IntMasterEnable();  // enable master interrupt
    SysTick_Init();  // configure SysTick Timer

    // Enable GPIOF
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOF));

    // Configure PF1 as output (Red LED)
    GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_1);

    // Configure PF4 as input (SW1)
    GPIOPinTypeGPIOInput(GPIO_PORTF_BASE, GPIO_PIN_4);
    GPIOPadConfigSet(GPIO_PORTF_BASE, GPIO_PIN_4, GPIO_STRENGTH_2MA, GPIO_PIN_TYPE_STD_WPU);

    while(1)
    {
        if (GPIOPinRead(GPIO_PORTF_BASE, GPIO_PIN_4) == 0)
        {
            GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1, GPIO_PIN_1);
        }
        else GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1, 0);
    }
}
