/*
 * sht21.c
 *
 *  Created on: Oct 13, 2025
 *      Author: controllerstech
 */


#include "sht21.h"
#include <string.h>

#define SHT21_I2C hi2c1
#define LOG_UART  huart1


#define SHT21_DELAY_MS(ms) HAL_Delay(ms)
extern I2C_HandleTypeDef SHT21_I2C;
extern UART_HandleTypeDef LOG_UART;

int _write(int fd, unsigned char *buf, int len) {
  if (fd == 1 || fd == 2) {                     // stdout or stderr ?
    HAL_UART_Transmit(&LOG_UART, buf, len, 999);  // Print to the UART
  }
  return len;
}

// CRC Check (Polynomial x^8 + x^5 + x^4 + 1 = 0x31)
static uint8_t SHT21_CheckCRC(uint8_t *data, uint8_t len, uint8_t checksum)
{
    uint8_t crc = 0x00;
    for (uint8_t i = 0; i < len; i++) {
        crc ^= data[i];
        for (uint8_t j = 0; j < 8; j++) {
            crc = (crc & 0x80) ? (crc << 1) ^ 0x31 : (crc << 1);
        }
    }
    return (crc == checksum);
}

// Initialization
SHT21_StatusTypeDef SHT21_Init()
{
    DEBUG_LOG("Initializing sensor...\r\n");
    SHT21_StatusTypeDef status = SHT21_SoftReset();

    if (status == SHT21_OK)
    	DEBUG_LOG("SHT21 initialized successfully.\r\n");
    else
    	DEBUG_LOG("SHT21 initialization failed.\r\n");
    return status;
}

// Soft Reset
SHT21_StatusTypeDef SHT21_SoftReset()
{
    uint8_t cmd = SHT21_SOFT_RESET;
    if (HAL_I2C_Master_Transmit(&SHT21_I2C, SHT21_I2C_ADDR, &cmd, 1, 100) != HAL_OK) {
    	DEBUG_LOG("Soft reset failed (I2C error)\r\n");
        return SHT21_ERROR;
    }

    SHT21_DELAY_MS(20);
    DEBUG_LOG("Soft reset done.\r\n");
    return SHT21_OK;
}

// User Register Read/Write
SHT21_StatusTypeDef SHT21_ReadUserRegister(uint8_t *value)
{
    uint8_t cmd = SHT21_READ_USER_REG;
    if (HAL_I2C_Master_Transmit(&SHT21_I2C, SHT21_I2C_ADDR, &cmd, 1, 100) != HAL_OK)
        return SHT21_ERROR;
    if (HAL_I2C_Master_Receive(&SHT21_I2C, SHT21_I2C_ADDR, value, 1, 100) != HAL_OK)
        return SHT21_ERROR;

    DEBUG_LOG("User register read: 0x%02X\r\n", *value);
    return SHT21_OK;
}

SHT21_StatusTypeDef SHT21_WriteUserRegister(uint8_t value)
{
    uint8_t buf[2] = {SHT21_WRITE_USER_REG, value};
    if (HAL_I2C_Master_Transmit(&SHT21_I2C, SHT21_I2C_ADDR, buf, 2, 100) != HAL_OK)
        return SHT21_ERROR;

    DEBUG_LOG("User register written: 0x%02X\r\n", value);
    return SHT21_OK;
}

// Internal raw data reader
static SHT21_StatusTypeDef SHT21_ReadRaw(uint8_t cmd, uint16_t *raw)
{
    uint8_t data[3];

    if (HAL_I2C_Master_Transmit(&SHT21_I2C, SHT21_I2C_ADDR, &cmd, 1, 100) != HAL_OK) {
    	DEBUG_LOG("I2C TX failed during command 0x%02X\r\n", cmd);
        return SHT21_ERROR;
    }

    SHT21_DELAY_MS(100);  // Typical max conversion time: 85 ms

    if (HAL_I2C_Master_Receive(&SHT21_I2C, SHT21_I2C_ADDR, data, 3, 100) != HAL_OK) {
    	DEBUG_LOG("I2C RX failed for command 0x%02X\r\n", cmd);
        return SHT21_ERROR;
    }

    if (!SHT21_CheckCRC(data, 2, data[2])) {
    	DEBUG_LOG("CRC mismatch for command 0x%02X\r\n", cmd);
        return SHT21_CRC_ERROR;
    }

    *raw = ((uint16_t)data[0] << 8) | data[1];
    *raw &= ~0x0003;  // Clear status bits
    return SHT21_OK;
}

// Temperature & Humidity
SHT21_StatusTypeDef SHT21_ReadTemperature(float *temperature)
{
    uint16_t raw;
    SHT21_StatusTypeDef status = SHT21_ReadRaw(SHT21_TRIG_TEMP_NOHOLD, &raw);
    if (status != SHT21_OK)
        return status;

    *temperature = -46.85f + 175.72f * (float)raw / 65536.0f;
    DEBUG_LOG("Temperature: %.2f C (raw: %u)\r\n", *temperature, raw);
    return SHT21_OK;
}

SHT21_StatusTypeDef SHT21_ReadHumidity(float *humidity)
{
    uint16_t raw;
    SHT21_StatusTypeDef status = SHT21_ReadRaw(SHT21_TRIG_HUM_NOHOLD, &raw);
    if (status != SHT21_OK)
        return status;

    *humidity = -6.0f + 125.0f * (float)raw / 65536.0f;
    if (*humidity > 100.0f) *humidity = 100.0f;
    if (*humidity < 0.0f) *humidity = 0.0f;

    DEBUG_LOG("Humidity: %.2f %%RH (raw: %u)\r\n", *humidity, raw);
    return SHT21_OK;
}
