#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/systick.h"
#include "driverlib/interrupt.h"

// System clock configuration
void systemClockConfig(void)
{
    SysCtlClockSet(SYSCTL_XTAL_16MHZ | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_SYSDIV_2_5);
}

// SysTick for delay
volatile uint32_t msTicks = 0;

void SysTick_Handler(void)
{
    msTicks++;
}

void delay_ms(uint32_t ms)
{
    uint32_t start = msTicks;
    while ((msTicks - start) < ms) {}
}

void SysTick_Init(void)
{
    SysTickPeriodSet(SysCtlClockGet() / 1000);
    SysTickIntRegister(SysTick_Handler);
    SysTickIntEnable();
    SysTickEnable();
}

// Flag for button press
volatile bool buttonPressed = false;

void GPIOF_Handler(void)
{
    GPIOIntClear(GPIO_PORTF_BASE, GPIO_PIN_4); // Clear interrupt flag
    buttonPressed = true;                        // Set flag
}

int main(void)
{
    systemClockConfig();
    SysTick_Init();

    // Enable GPIOF
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOF));

    // Configure pins
    GPIOPinTypeGPIOInput(GPIO_PORTF_BASE, GPIO_PIN_4);
    GPIOPadConfigSet(GPIO_PORTF_BASE, GPIO_PIN_4, GPIO_STRENGTH_2MA, GPIO_PIN_TYPE_STD_WPU);

    GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_1);

    // Setup GPIO interrupt
    GPIOIntDisable(GPIO_PORTF_BASE, GPIO_PIN_4);
    GPIOIntClear(GPIO_PORTF_BASE, GPIO_PIN_4);
    GPIOIntTypeSet(GPIO_PORTF_BASE, GPIO_PIN_4, GPIO_FALLING_EDGE);
    GPIOIntRegister(GPIO_PORTF_BASE, GPIOF_Handler);
    GPIOIntEnable(GPIO_PORTF_BASE, GPIO_PIN_4);
    
    // Enable Master Interrupt
    IntMasterEnable();

    // Main loop
    while(1)
    {
        static bool ledState = false;

        if(buttonPressed)
        {
            ledState = !ledState; // Change LED state
            GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1, ledState ? GPIO_PIN_1 : 0);
            delay_ms(500);  // Prevent Debouncing
            buttonPressed = false;
        }
    }
}
