/*
  ***************************************************************************************************************
  ***************************************************************************************************************
  ***************************************************************************************************************

  File:		  ESP8266_STM32.c
  Author:     ControllersTech.com
  Updated:    Sep 6, 2025

  ***************************************************************************************************************
  Copyright (C) 2017 ControllersTech.com

  This is a free software under the GNU license, you can redistribute it and/or modify it under the terms
  of the GNU General Public License version 3 as published by the Free Software Foundation.
  This software library is shared with public for educational purposes, without WARRANTY and Author is not liable for any damages caused directly
  or indirectly by this software, read more about this on the GNU General Public License.

  ***************************************************************************************************************
*/

#include "ESP8266_STM32.h"
#include "stdlib.h"

ESP8266_ConnectionState ESP_ConnState = ESP8266_DISCONNECTED;   // Default state

static char esp_rx_buffer[2048];
static ESP8266_Status ESP_GetIP(char *ip_buffer, uint16_t buffer_len);
static ESP8266_Status ESP_SendCommand(char *cmd, const char *ack, uint32_t timeout);
static ESP8266_Status ESP_SendBinary(uint8_t *bin, size_t len, const char *ack, uint32_t timeout);
static int MQTT_BuildConnect(uint8_t *packet, const char *clientID, const char *username, const char *password, uint16_t keepalive);
static uint16_t ESP_DMA_GetWritePos(void);
static void ESP_DMA_Flush(void);

/* ===================== DMA CHANGE ===================== */
#define ESP_DMA_RX_BUF_SIZE 1024
static uint8_t esp_dma_rx_buf[ESP_DMA_RX_BUF_SIZE];
static volatile uint16_t esp_dma_rx_head = 0;

ESP8266_Status ESP_Init(void)
{
	ESP8266_Status res;
	USER_LOG("Initializing ESP8266...");
	HAL_Delay(1000);

//	res = ESP_SendCommand("AT+RST\r\n", "OK", 2000);
//    if (res != ESP8266_OK){
//    	DEBUG_LOG("Failed to Reset ESP8266...");
//    	return res;
//    }
//
//    USER_LOG("Waiting 5 Seconds for Reset to Complete...");
//    HAL_Delay(5000);  // wait for reset to complete

    // Start DMA RX
    HAL_UART_Receive_DMA(&ESP_UART, esp_dma_rx_buf, ESP_DMA_RX_BUF_SIZE);
    ESP_DMA_Flush();

    res = ESP_SendCommand("AT\r\n", "OK", 2000);
    if (res != ESP8266_OK){
    	DEBUG_LOG("ESP8266 Not Responding...");
    	return res;
    }

    res = ESP_SendCommand("ATE0\r\n", "OK", 2000); // Disable echo
    if (res != ESP8266_OK){
    	DEBUG_LOG("Disable echo Command Failed...");
    	return res;
    }


    USER_LOG("ESP8266 Initialized Successfully...");
    return ESP8266_OK;
}

ESP8266_Status ESP_CheckTCPConnection(void)
{
    return ESP_SendCommand("AT+CIPSTATUS\r\n", "STATUS:3", 2000);
}

ESP8266_Status ESP_ConnectWiFi(const char *ssid, const char *password, char *ip_buffer, uint16_t buffer_len)
{
    USER_LOG("Setting in Station Mode");
    // Set in Station Mode
    char cmd[128];
    snprintf(cmd, sizeof(cmd), "AT+CWMODE=1\r\n");

    ESP8266_Status result = ESP_SendCommand(cmd, "OK", 2000); // wait up to 2s
    if (result != ESP8266_OK)
    {
    	USER_LOG("Station Mode Failed.");
        return result;
    }

    USER_LOG("Connecting to WiFi SSID: %s", ssid);
    // Send join command
    snprintf(cmd, sizeof(cmd), "AT+CWJAP=\"%s\",\"%s\"\r\n", ssid, password);

    result = ESP_SendCommand(cmd, "WIFI CONNECTED", 10000); // wait up to 10s
    if (result != ESP8266_OK)
    {
    	USER_LOG("WiFi connection failed.");
        ESP_ConnState = ESP8266_DISCONNECTED;
        return result;
    }

    USER_LOG("WiFi Connected. Waiting for IP...");
    ESP_ConnState = ESP8266_CONNECTED_NO_IP;
    // Fetch IP with retries inside ESP_GetIP
    result = ESP_GetIP(ip_buffer, buffer_len);
    if (result != ESP8266_OK)
    {
    	USER_LOG("Failed to fetch IP. Status=%d", result);
        return result;
    }

    USER_LOG("WiFi + IP ready: %s", ip_buffer);
    return ESP8266_OK;
}


ESP8266_ConnectionState ESP_GetConnectionState(void)
{
    return ESP_ConnState;
}

/* ===================== ThingSpeak IMPLEMENTATION ===================== */

ESP8266_Status ESP_SendToThingSpeak(const char *apiKey, float val1, float val2, float val3)
{
    char cmd[256];
    ESP8266_Status result;
    static uint16_t tail = 0; // Track DMA buffer position
        size_t parse_idx = 0;

        USER_LOG("Connecting to ThingSpeak...");

        // Clear esp_rx_buffer and reset parse_idx
        memset(esp_rx_buffer, 0, sizeof(esp_rx_buffer));
        parse_idx = 0;
        ESP_DMA_Flush();

    // 1. Start TCP connection (HTTP port 80)
    snprintf(cmd, sizeof(cmd), "AT+CIPSTART=\"TCP\",\"api.thingspeak.com\",80\r\n");
    result = ESP_SendCommand(cmd, "CONNECT", 5000);
    if (result != ESP8266_OK) {
        USER_LOG("TCP connection to ThingSpeak failed.");
        return result;
    }

    // 2. Build HTTP GET request
    char httpReq[256];
    snprintf(httpReq, sizeof(httpReq),
             "GET /update?api_key=%s&field1=%.2f&field2=%.2f&field3=%.2f\r\n", apiKey, val1, val2, val3);

    // 3. Tell ESP how many bytes we will send
    snprintf(cmd, sizeof(cmd), "AT+CIPSEND=%d\r\n", (int)strlen(httpReq));
    result = ESP_SendCommand(cmd, ">", 2000);
    if (result != ESP8266_OK) {
        USER_LOG("CIPSEND failed.");
        return result;
    }

    // 4. Send actual request and wait for ThingSpeak response
    result = ESP_SendCommand(httpReq, "SEND OK", 5000);
    if (result != ESP8266_OK) {
        USER_LOG("Failed to send HTTP request.");
        return result;
    }

    // 5. Parse ThingSpeak reply in esp_rx_buffer

    // Accumulate data from DMA buffer into esp_rx_buffer
        uint16_t head = ESP_DMA_GetWritePos();
        while (tail != head) {
            if (parse_idx >= sizeof(esp_rx_buffer) - 1) {
                DEBUG_LOG("esp_rx_buffer overflow");
                memset(esp_rx_buffer, 0, sizeof(esp_rx_buffer));
                tail = head; // Reset tail to skip unprocessed data
                return ESP8266_ERROR;
            }
            esp_rx_buffer[parse_idx++] = esp_dma_rx_buf[tail++];
            if (tail >= ESP_DMA_RX_BUF_SIZE) tail = 0;
            esp_rx_buffer[parse_idx] = '\0';
        }

        // Log esp_rx_buffer for debugging
        if (parse_idx > 0) {
            DEBUG_LOG("\nesp_rx_buffer text: %s", esp_rx_buffer);
        }

        // Parse all +IPD messages in esp_rx_buffer
        char *ipd = esp_rx_buffer;
        while (ipd && *ipd) {
            ipd = strstr(ipd, "+IPD,");
            if (!ipd) break;

            DEBUG_LOG("Found +IPD: %s", ipd);
            char *colon = strchr(ipd, ':');
            if (!colon) {
            	DEBUG_LOG("Incomplete +IPD header");
                return ESP8266_NO_RESPONSE;
            }

            DEBUG_LOG("Colon found at offset: %d", colon - ipd);
            char *len_str = ipd + 5; // Skip "+IPD,"
            uint32_t ipd_len = atoi(len_str);
            char *data = colon + 1;

            // Check if enough data is available
            if ((colon - esp_rx_buffer) + 1 + ipd_len > parse_idx) {
            	DEBUG_LOG("Incomplete +IPD data");
                return ESP8266_NO_RESPONSE;
            }

            // Parse entry ID
            int entryId = atoi(data);
            DEBUG_LOG("ThingSpeak entry ID: %d", entryId);

            if (entryId > 0) {
                USER_LOG("Update successful! Entry ID: %d", entryId);
                // Shift remaining data
                int parsed_len = (colon - esp_rx_buffer) + 1 + ipd_len;
                int leftover = parse_idx - parsed_len;
                if (leftover > 0) {
                    memmove(esp_rx_buffer, esp_rx_buffer + parsed_len, leftover);
                    parse_idx = leftover;
                } else {
                    parse_idx = 0;
                }
                memset(esp_rx_buffer + parse_idx, 0, sizeof(esp_rx_buffer) - parse_idx);
                return ESP8266_OK;
            }

            DEBUG_LOG("ThingSpeak returned invalid entry ID.");
            // Skip this +IPD
            int skip_len = (colon - esp_rx_buffer) + 1 + ipd_len;
            int leftover = parse_idx - skip_len;
            if (leftover > 0) {
                memmove(esp_rx_buffer, esp_rx_buffer + skip_len, leftover);
                parse_idx = leftover;
            } else {
                parse_idx = 0;
            }
            memset(esp_rx_buffer + parse_idx, 0, sizeof(esp_rx_buffer) - parse_idx);
            ipd = esp_rx_buffer;
        }
    USER_LOG("No valid ThingSpeak response found.");
    return ESP8266_ERROR;
}

/* ===================== MQTT IMPLEMENTATION ===================== */

ESP8266_Status ESP_MQTT_Connect(const char *broker, uint16_t port, const char *clientID,
		const char *username, const char *password, uint16_t keepalive)
{
    char cmd[64];
    uint8_t packet[256];
    int len = 0;
    ESP8266_Status res;

    USER_LOG("Connecting to MQTT broker %s:%d", broker, port);

    /****** Step 1: TCP connect ******/

    snprintf(cmd, sizeof(cmd), "AT+CIPSTART=\"TCP\",\"%s\",%d\r\n", broker, port);
    res = ESP_SendCommand(cmd, "CONNECT", 5000);
    if (res != ESP8266_OK){
    	 DEBUG_LOG("CIPSTART Failed with ESP ERROR %d..", res);
    	 return res;
    }

    /****** Step 2: Build MQTT CONNECT packet ******/

    len = MQTT_BuildConnect(packet, clientID, username, password, keepalive);

    /****** Step 3: Tell ESP how many bytes to send ******/

    snprintf(cmd, sizeof(cmd), "AT+CIPSEND=%d\r\n", len);
    res = ESP_SendCommand(cmd, ">", 2000);
    if (res != ESP8266_OK){
    	DEBUG_LOG("CIPSEND Failed with ESP ERROR %d..", res);
    	 return res;
    }

    /****** Step 4: Send packet and wait for CONNACK ******/

    res = ESP_SendBinary(packet, len, "\x20\x02\x00\x00", 10000);
    if (res != ESP8266_OK){
    	DEBUG_LOG("Send Connect Command Failed with ESP ERROR %d..", res);
    	USER_LOG("MQTT CONNACK failed.");
    	 return res;
    }

	USER_LOG("MQTT CONNACK received, broker accepted connection.");
    return ESP8266_OK;
}

ESP8266_Status ESP_MQTT_Publish(const char *topic, const char *message, uint8_t qos)
{
    char cmd[64];
    uint8_t packet[256];
    int len = 0;
    ESP8266_Status res;

    USER_LOG("Publishing to MQTT Topic:message %s:%s", topic, message);

    /****** Step 1: Build MQTT Publish packet ******/

    /* Fixed Header */
    packet[len++] = 0x30 | (qos << 1); // PUBLISH, QoS
    int remLenPos = len++;  // will be calculated later

    /* Variable Header */
    // Topic
    uint16_t tlen = strlen(topic);
    packet[len++] = tlen >> 8;  // store topic len
    packet[len++] = tlen & 0xFF;  // store topic len
    memcpy(&packet[len], topic, tlen); len += tlen;  // store topic

    /* Payload */
    // Message
    uint16_t mlen = strlen(message);
    memcpy(&packet[len], message, mlen); len += mlen;  // store message

    // Remaining length
    packet[remLenPos] = len - 2;  // remove first 2 bytes of Fixed Header

    /****** Step 2: Tell ESP how many bytes to send  ******/

    snprintf(cmd, sizeof(cmd), "AT+CIPSEND=%d\r\n", len);
    res = ESP_SendCommand(cmd, ">", 2000);
    if (res != ESP8266_OK){
    	DEBUG_LOG("CIPSEND Failed with ESP ERROR %d..", res);
    	 return res;
    }

    /****** Step 3: Send packet and wait for ACK ******/

    res = ESP_SendBinary(packet, len, "SEND OK", 5000);
    if (res != ESP8266_OK){
    	DEBUG_LOG("Publish Command Failed with ESP ERROR %d..", res);
    	 return res;
    }

    USER_LOG ("Successfully Published to Broker..");
    return ESP8266_OK;
}

ESP8266_Status ESP_MQTT_Ping(void)
{
    char cmd[32];
    ESP8266_Status res;
    uint8_t packet[2];

    USER_LOG("Sending PINGREQ");

    /****** Step 1: Build MQTT Publish packet ******/
    /* Fixed Header */
    packet[0] = 0xC0; 	packet[1] = 0x00;  // PINGREQ

    /****** Step 2: Tell ESP how many bytes to send  ******/
    snprintf(cmd, sizeof(cmd), "AT+CIPSEND=2\r\n");
    res = ESP_SendCommand(cmd, ">", 2000);
    if (res != ESP8266_OK){
    	 DEBUG_LOG("CIPSEND Failed with ESP ERROR %d..", res);
    	 return res;
    }

    /****** Step 3: Send packet and wait for PINGRESP (0xD0 0x00) ******/
    res = ESP_SendBinary(packet, 2, "\xD0", 2000);
    if (res != ESP8266_OK){
    	 DEBUG_LOG("Publish Command Failed with ESP ERROR %d..", res);
    	 return res;
    }

    USER_LOG("PINGREQ Sucessful");
    return ESP8266_OK;
}


/* ===================== MQTT SUBSCRIBE ===================== */
ESP8266_Status ESP_MQTT_Subscribe(const char *topic, uint8_t qos)
{
    char cmd[64];
    uint8_t packet[256];
    int len = 0;
    ESP8266_Status res;

    USER_LOG("Subscribing to MQTT Topic: %s", topic);

    /****** Step 1: Build MQTT Publish packet ******/
    /* Fixed Header */
    packet[len++] = 0x82;   // SUBSCRIBE
    int remLenPos = len++;

    /* Variable Header: Packet Identifier */
    packet[len++] = 0x00;
    packet[len++] = 0x01;

    /* Payload: Topic */
    uint16_t tlen = strlen(topic);
    packet[len++] = tlen >> 8;
    packet[len++] = tlen & 0xFF;
    memcpy(&packet[len], topic, tlen); len += tlen;
    packet[len++] = qos;  // Requested QoS

    // Remaining length
    packet[remLenPos] = len - 2;  // remove first 2 bytes of Fixed Header

    /****** Step 2: Tell ESP how many bytes to send  ******/
    snprintf(cmd, sizeof(cmd), "AT+CIPSEND=%d\r\n", len);
    res = ESP_SendCommand(cmd, ">", 2000);
    if (res != ESP8266_OK) return res;

    /****** Step 3: Send packet and wait for PINGRESP (0xD0 0x00) ******/
    res = ESP_SendBinary(packet, len, "\x90", 5000); // SUBACK = 0x90
    if (res != ESP8266_OK) return res;

    USER_LOG("Successfully Subscribed to Broker..");
    return ESP8266_OK;
}

/* ===================== MQTT INCOMING (DMA) ===================== */
ESP8266_Status ESP_MQTT_HandleIncoming(char *topic_buffer, uint16_t topic_buf_len, char *msg_buffer, uint16_t msg_buf_len)
{
    static uint16_t tail = 0;
    uint16_t head = ESP_DMA_GetWritePos();

    static char parse_buf[512];
    static uint16_t parse_idx = 0;

    // Accumulate data from DMA buffer
    while (tail != head) {
        if (parse_idx >= sizeof(parse_buf) - 1) {
        	DEBUG_LOG("parse_buf overflow");  // If parse_buf Overflowed
            parse_idx = 0;
            memset(parse_buf, 0, sizeof(parse_buf));  // clear the buffer so to load new data
            return ESP8266_ERROR;
        }
        parse_buf[parse_idx++] = esp_dma_rx_buf[tail++];  // update the parse_buf with data from DMA buf
        if (tail >= ESP_DMA_RX_BUF_SIZE) tail = 0;
        parse_buf[parse_idx] = '\0';
    }

    // Process all +IPD messages in parse_buf
    char *ipd = parse_buf;
    while (ipd && *ipd) {

        ipd = strstr(ipd, "+IPD,");  // search for IPD
        if (!ipd) return ESP8266_NO_RESPONSE;  // NO IPD, exit

        // IPD Found
        DEBUG_LOG("Found +IPD: %s", ipd);
        char *colon = strchr(ipd, ':');  // search for :
        if (!colon) return ESP8266_NO_RESPONSE;  // No :, exit

        // : Found
        DEBUG_LOG("Colon found at offset: %d", colon - ipd);
        // Parse +IPD,<len>
        char *len_str = ipd + 5; // Skip "+IPD,"
        uint32_t ipd_len = atoi(len_str);
        uint8_t *mqtt = (uint8_t *)(colon + 1);

        // Check if enough data is available
        if ((colon - parse_buf) + 1 + ipd_len > parse_idx) {
            DEBUG_LOG("Incomplete MQTT packet, waiting for more data");
            return ESP8266_NO_RESPONSE;
        }

        // Check for MQTT PUBLISH
        if ((mqtt[0] & 0xF0) == 0x30) {  // if it is a publish message
            // Decode variable-length remaining length
            uint32_t rem_len = 0;
            int multiplier = 1;
            int index = 1;
            do {
                if (index >= ipd_len) {
                    DEBUG_LOG("Incomplete remaining length");
                    goto skip_packet;
                }
                rem_len += (mqtt[index] & 0x7F) * multiplier;
                multiplier *= 128;
                if (multiplier > 128*128*128) {
                    DEBUG_LOG("Invalid remaining length");
                    goto skip_packet;
                }
                index++;
            } while ((mqtt[index - 1] & 0x80) != 0);

            // Check QoS level (bits 1-2 of mqtt[0])
            uint8_t qos = (mqtt[0] >> 1) & 0x03;

            // Parse topic length and topic
            if (index + 2 > ipd_len) {
            	DEBUG_LOG("Incomplete topic length");
                goto skip_packet;
            }
            uint16_t tlen = (mqtt[index] << 8) | mqtt[index + 1];
            index += 2;

            if (tlen >= topic_buf_len || index + tlen > ipd_len) {
            	DEBUG_LOG("Topic too long or incomplete: %u", tlen);
                goto skip_packet;
            }
            memcpy(topic_buffer, &mqtt[index], tlen);
            topic_buffer[tlen] = '\0';
            index += tlen;

            // Skip packet identifier for QoS 1 or 2
            if (qos > 0) {
                if (index + 2 > ipd_len) {
                	DEBUG_LOG("Incomplete packet identifier");
                    goto skip_packet;
                }
                index += 2; // Skip 2-byte packet identifier
            }

            // Parse message
            uint32_t mlen = rem_len - (2 + tlen + (qos > 0 ? 2 : 0));
            if (mlen >= msg_buf_len || index + mlen > ipd_len) {
            	DEBUG_LOG("Message too long or incomplete: %lu", mlen);
                goto skip_packet;
            }
            memcpy(msg_buffer, &mqtt[index], mlen);
            msg_buffer[mlen] = '\0';

            USER_LOG("MQTT Msg -> Topic: %s, Message: %s", topic_buffer, msg_buffer);

            // Shift remaining data
            int parsed_len = (colon - parse_buf) + 1 + ipd_len;
            int leftover = parse_idx - parsed_len;
            if (leftover > 0) {
                memmove(parse_buf, parse_buf + parsed_len, leftover);
                parse_idx = leftover;
            } else {
                parse_idx = 0;
            }
            memset(parse_buf + parse_idx, 0, sizeof(parse_buf) - parse_idx);
            return ESP8266_OK;
        } else {
        	DEBUG_LOG("Not a PUBLISH packet: 0x%02X", mqtt[0]);
        }

    skip_packet:
        // Skip this +IPD packet
        int skip_len = (colon - parse_buf) + 1 + ipd_len;
        int leftover = parse_idx - skip_len;
        if (leftover > 0) {
            memmove(parse_buf, parse_buf + skip_len, leftover);
            parse_idx = leftover;
        } else {
            parse_idx = 0;
        }
        memset(parse_buf + parse_idx, 0, sizeof(parse_buf) - parse_idx);
        ipd = parse_buf; // Continue searching for next +IPD
    }

    return ESP8266_NO_RESPONSE;
}

/* ===================== Static Functions ===================== */
static ESP8266_Status ESP_GetIP(char *ip_buffer, uint16_t buffer_len)
{
    DEBUG_LOG("Fetching IP Address...");

    for (int attempt = 1; attempt <= 3; attempt++)
    {
        ESP_DMA_Flush();  // clear old data

        // Send CIFSR command
        ESP8266_Status result = ESP_SendCommand("AT+CIFSR\r\n", "OK", 5000);
        if (result != ESP8266_OK)
        {
            DEBUG_LOG("CIFSR failed on attempt %d", attempt);
            continue;
        }

        // Now esp_rx_buffer should contain +CIFSR output
        char *search = esp_rx_buffer;
        char *last_ip = NULL;

        while ((search = strstr(search, "STAIP,\"")) != NULL)
        {
            search += 7; // move past STAIP,"
            char *end = strchr(search, '"');
            if (end && ((end - search) < buffer_len))
            {
                last_ip = search;
                int len = end - search;
                strncpy(ip_buffer, search, len);
                ip_buffer[len] = '\0';
            }
            search++; // keep scanning
        }

        if (last_ip)
        {
            if (strcmp(ip_buffer, "0.0.0.0") == 0)
            {
                DEBUG_LOG("Attempt %d: IP not ready yet (0.0.0.0). Retrying...", attempt);
                ESP_ConnState = ESP8266_CONNECTED_NO_IP;
                HAL_Delay(1000);
                continue;
            }

            DEBUG_LOG("Got IP: %s", ip_buffer);
            ESP_ConnState = ESP8266_CONNECTED_IP;
            return ESP8266_OK;
        }

        DEBUG_LOG("Attempt %d: Failed to parse STAIP.", attempt);
        HAL_Delay(500);
    }

    DEBUG_LOG("Failed to fetch IP after retries.");
    ESP_ConnState = ESP8266_CONNECTED_NO_IP;
    return ESP8266_ERROR;
}
static ESP8266_Status ESP_SendCommand(char *cmd, const char *ack, uint32_t timeout)
{
    uint32_t tickstart = HAL_GetTick();
    ESP_DMA_Flush();

    if (strlen(cmd) > 0) {
        DEBUG_LOG("Sending: %s", cmd);
        if (HAL_UART_Transmit(&ESP_UART, (uint8_t*)cmd, strlen(cmd), HAL_MAX_DELAY) != HAL_OK)
            return ESP8266_ERROR;
    }

    while ((HAL_GetTick() - tickstart) < timeout)
    {
        uint16_t head = ESP_DMA_GetWritePos();
        while (esp_dma_rx_head != head)
        {
            char c = esp_dma_rx_buf[esp_dma_rx_head++];
            if (esp_dma_rx_head >= ESP_DMA_RX_BUF_SIZE) esp_dma_rx_head = 0;

            size_t len = strlen(esp_rx_buffer);
            if (len < sizeof(esp_rx_buffer) - 1) {
                esp_rx_buffer[len] = c;
                esp_rx_buffer[len+1] = '\0';
            }

            if (strstr(esp_rx_buffer, ack)) {
                DEBUG_LOG("Matched ACK: %s", ack);
                return ESP8266_OK;
            }
        }
    }

    DEBUG_LOG("Timeout or no ACK. Buffer: %s", esp_rx_buffer);
    return ESP8266_TIMEOUT;
}

static ESP8266_Status ESP_SendBinary(uint8_t *bin, size_t len, const char *ack, uint32_t timeout)
{
    uint32_t tickstart = HAL_GetTick();
    ESP_DMA_Flush();

    DEBUG_LOG("Sending Binary Packet");
    if (HAL_UART_Transmit(&ESP_UART, bin, len, HAL_MAX_DELAY) != HAL_OK){
        return ESP8266_ERROR;
    }

    while ((HAL_GetTick() - tickstart) < timeout)
    {
        uint16_t head = ESP_DMA_GetWritePos();
        while (esp_dma_rx_head != head)
        {
            char c = esp_dma_rx_buf[esp_dma_rx_head++];
            if (esp_dma_rx_head >= ESP_DMA_RX_BUF_SIZE) esp_dma_rx_head = 0;

            size_t buflen = strlen(esp_rx_buffer);
            if (buflen < sizeof(esp_rx_buffer) - 1) {
                esp_rx_buffer[buflen] = c;
                esp_rx_buffer[buflen+1] = '\0';
            }

            if (strstr(esp_rx_buffer, ack)) {
                DEBUG_LOG("Matched ACK: %s", ack);
                return ESP8266_OK;
            }

            if (strstr(esp_rx_buffer, "ERROR")) {
                DEBUG_LOG("ESP Disconnected or ERROR");
                return ESP8266_ERROR;
            }
        }
    }

    DEBUG_LOG("Timeout or no ACK. Buffer: %s", esp_rx_buffer);
    return ESP8266_TIMEOUT;
}

static int MQTT_BuildConnect(uint8_t *packet, const char *clientID, const char *username, const char *password, uint16_t keepalive)
{
    int len = 0;
    /* Fixed Header */
    packet[len++] = 0x10;   // CONNECT packet type
    int remLenPos = len++;  // Remaining length placeholder

    /* Variable Header */
    packet[len++] = 0x00; packet[len++] = 0x04;
    packet[len++] = 'M'; packet[len++] = 'Q'; packet[len++] = 'T'; packet[len++] = 'T';
    packet[len++] = 0x04;   // Protocol Level = 4 (MQTT 3.1.1)

    uint8_t connectFlags = 0x02; // Clean Session
    if (username) connectFlags |= 0x80;
    if (password) connectFlags |= 0x40;
    packet[len++] = connectFlags;

    // Keep Alive
    packet[len++] = (keepalive >> 8) & 0xFF;
    packet[len++] = (keepalive & 0xFF);

    /* Payload */
    // Client ID
    uint16_t cid_len = strlen(clientID);
    packet[len++] = cid_len >> 8;
    packet[len++] = cid_len & 0xFF;
    memcpy(&packet[len], clientID, cid_len); len += cid_len;

    // Username
    if (username) {
        uint16_t ulen = strlen(username);
        packet[len++] = ulen >> 8;
        packet[len++] = ulen & 0xFF;
        memcpy(&packet[len], username, ulen); len += ulen;
    }

    // Password
    if (password) {
        uint16_t plen = strlen(password);
        packet[len++] = plen >> 8;
        packet[len++] = plen & 0xFF;
        memcpy(&packet[len], password, plen); len += plen;
    }

    // Remaining length from Fixed Header
    packet[remLenPos] = len - 2;  // remove first 2 bytes of Fixed Header
    return len;
}

static uint16_t ESP_DMA_GetWritePos(void)
{
    return ESP_DMA_RX_BUF_SIZE - __HAL_DMA_GET_COUNTER(ESP_UART.hdmarx);
}

static void ESP_DMA_Flush(void)
{
    esp_dma_rx_head = ESP_DMA_GetWritePos();  // discard old data
    memset(esp_rx_buffer, 0, sizeof(esp_rx_buffer));
}
